<?php
/**
 * VIP-based rebate system functions
 * Only users with VIP Level 1 or higher get rebates
 * Rebate rates are based on VIP levels
 */

/**
 * Create bet_rebates table if not exists
 */
function createBetRebatesTable($conn) {
    $sql = "CREATE TABLE IF NOT EXISTS bet_rebates (
        id INT AUTO_INCREMENT PRIMARY KEY,
        user_id INT NOT NULL,
        bet_id INT NOT NULL,
        game_table VARCHAR(50) NOT NULL,
        bet_amount DECIMAL(10,2) NOT NULL,
        rebate_rate DECIMAL(10,6) NOT NULL,
        rebate_amount DECIMAL(10,2) NOT NULL,
        vip_level INT NOT NULL DEFAULT 0,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        UNIQUE KEY unique_bet_rebate (user_id, bet_id, game_table),
        INDEX idx_user_id (user_id),
        INDEX idx_game_table (game_table),
        INDEX idx_created_at (created_at)
    )";
    
    $conn->query($sql);
}

/**
 * Get user's VIP level
 * @param mysqli $conn Database connection
 * @param int $userId User ID
 * @return int VIP level (0 if not VIP)
 */
function getUserVipLevel($conn, $userId) {
    $vipQuery = "SELECT lvl FROM vip WHERE userid = '$userId'";
    $vipResult = $conn->query($vipQuery);
    
    if ($vipResult && $vipResult->num_rows > 0) {
        $vipRow = $vipResult->fetch_assoc();
        return (int)$vipRow['lvl'];
    }
    
    return 0; // No VIP level
}

/**
 * Get rebate rate based on VIP level
 * @param int $vipLevel VIP level
 * @return float Rebate rate
 */
function getVipRebateRate($vipLevel) {
    $vipRates = [
        1 => 0.0005,  // 0.05%
        2 => 0.0005,  // 0.05%
        3 => 0.001,   // 0.1%
        4 => 0.001,   // 0.1%
        5 => 0.001,   // 0.1%
        6 => 0.0015,  // 0.15%
        7 => 0.0015,  // 0.15%
        8 => 0.0015,  // 0.15%
        9 => 0.002,   // 0.2%
        10 => 0.003   // 0.3%
    ];
    
    return $vipRates[$vipLevel] ?? 0; // Return 0 if not VIP
}

/**
 * Process rebate for a bet (VIP-based)
 * @param mysqli $conn Database connection
 * @param int $userId User ID
 * @param int $betId Bet ID
 * @param string $gameTable Game table name
 * @param float $betAmount Bet amount
 * @param string $betTime Bet timestamp
 * @return bool Success status
 */
function processBetRebate($conn, $userId, $betId, $gameTable, $betAmount, $betTime) {
    // Get user's VIP level
    $vipLevel = getUserVipLevel($conn, $userId);
    
    // Only process rebate if user is VIP level 1 or higher
    if ($vipLevel < 1) {
        return true; // No rebate for non-VIP users
    }
    
    // Get rebate rate based on VIP level
    $rebateRate = getVipRebateRate($vipLevel);
    
    if ($rebateRate == 0) {
        return true; // No rebate for this VIP level
    }
    
    // Calculate rebate amount
    $rebateAmount = $betAmount * $rebateRate;
    
    // Check if rebate already exists for this bet
    $checkQuery = "SELECT COUNT(*) as count FROM bet_rebates WHERE user_id = '$userId' AND bet_id = '$betId' AND game_table = '$gameTable'";
    $checkResult = $conn->query($checkQuery);
    $checkRow = $checkResult->fetch_assoc();
    
    if ($checkRow['count'] == 0) {
        // Insert rebate record
        $insertQuery = "INSERT INTO bet_rebates (user_id, bet_id, game_table, bet_amount, rebate_rate, rebate_amount, vip_level, created_at) 
                       VALUES ('$userId', '$betId', '$gameTable', '$betAmount', '$rebateRate', '$rebateAmount', '$vipLevel', '$betTime')";
        
        if ($conn->query($insertQuery)) {
            return true;
        }
    }
    
    return true; // Already processed or failed
}

/**
 * Get user rebate information for API response
 * @param mysqli $conn Database connection
 * @param int $userId User ID
 * @return array Rebate information
 */
function getUserRebateInfo($conn, $userId) {
    $today = date('Y-m-d');
    
    // Get user's current VIP level
    $vipLevel = getUserVipLevel($conn, $userId);
    $currentRebateRate = getVipRebateRate($vipLevel);
    
    // Get rebate totals
    $rebateQuery = "SELECT 
                    SUM(rebate_amount) as total_rebate,
                    SUM(CASE WHEN DATE(created_at) = '$today' THEN rebate_amount ELSE 0 END) as day_rebate
                    FROM bet_rebates 
                    WHERE user_id = '$userId'";
    $rebateResult = $conn->query($rebateQuery);
    $rebateData = $rebateResult->fetch_assoc();
    
    $totalRebateAmount = $rebateData['total_rebate'] ?? 0;
    $dayRebateAmount = $rebateData['day_rebate'] ?? 0;
    
    // Get individual bet records for wash list
    $washList = [];
    $washListQuery = "SELECT bet_id, bet_amount, rebate_rate, rebate_amount, created_at, game_table, vip_level 
                      FROM bet_rebates 
                      WHERE user_id = '$userId' 
                      ORDER BY created_at DESC 
                      LIMIT 50";
    $washListResult = $conn->query($washListQuery);
    
    if ($washListResult && $washListResult->num_rows > 0) {
        while ($washRow = $washListResult->fetch_assoc()) {
            $washList[] = [
                'washVolume' => (float)$washRow['bet_amount'],
                'washRate' => (float)$washRow['rebate_rate'],
                'rebateAmount' => (float)$washRow['rebate_amount'],
                'gameTable' => $washRow['game_table'],
                'betTime' => $washRow['created_at'],
                'betId' => $washRow['bet_id'],
                'vipLevel' => (int)$washRow['vip_level']
            ];
        }
    }
    
    return [
        'codeWashAmount' => 0,
        'dayRebate' => (float)$dayRebateAmount,
        'totalRebate' => (float)$totalRebateAmount,
        'washRate' => $currentRebateRate,
        'washList' => $washList,
        'vipLevel' => $vipLevel
    ];
}

/**
 * Process historical bets for rebate calculation (for GetCodeWashAmount.php)
 * @param mysqli $conn Database connection
 * @param int $userId User ID
 * @return array Rebate information
 */
function processHistoricalBetsForRebate($conn, $userId) {
    // Get user's current VIP level
    $vipLevel = getUserVipLevel($conn, $userId);
    
    // Only process if user is VIP level 1 or higher
    if ($vipLevel < 1) {
        return [
            'codeWashAmount' => 0,
            'dayRebate' => 0,
            'totalRebate' => 0,
            'washRate' => 0,
            'washList' => [],
            'vipLevel' => 0
        ];
    }
    
    // Game tables to process
    $gameTables = [
        'bajikattuttate',
        'bajikattuttate_kemuru',
        'bajikattuttate_kemuru_drei',
        'bajikattuttate_kemuru_funf',
        'bajikattuttate_kemuru_zehn',
        'bajikattuttate_aidudi',
        'bajikattuttate_aidudi_drei',
        'bajikattuttate_aidudi_funf',
        'bajikattuttate_aidudi_zehn',
        'bajikattuttate_trx',
        'bajikattuttate_trx_drei',
        'bajikattuttate_trx_funf',
        'bajikattuttate_trx_zehn'
    ];
    
    foreach ($gameTables as $table) {
        // Check if table exists
        $tableCheck = $conn->query("SHOW TABLES LIKE '$table'");
        if ($tableCheck->num_rows == 0) {
            continue;
        }
        
        // Get user bets from this table
        $betsQuery = "SELECT aikona, ketebida, tiarikala FROM $table WHERE byabaharkarta = '$userId' ORDER BY aikona DESC LIMIT 100";
        $betsResult = $conn->query($betsQuery);
        
        if ($betsResult && $betsResult->num_rows > 0) {
            while ($betRow = $betsResult->fetch_assoc()) {
                $betId = $betRow['aikona'];
                $betAmount = $betRow['ketebida'];
                $betTime = $betRow['tiarikala'];
                
                // Process rebate for this bet
                processBetRebate($conn, $userId, $betId, $table, $betAmount, $betTime);
            }
        }
    }
    
    // Return updated rebate info
    return getUserRebateInfo($conn, $userId);
}
?>
